﻿using System;
using System.Collections.ObjectModel;
using System.Diagnostics;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.ServiceModel.Description;
using System.ServiceModel.Dispatcher;
using System.Security;
using System.Collections.Generic;
using BMS.Facade.Fault;
using System.Text;
using BMS.FaultContracts;
using BMS.Utils.Properties;
using BMS.Utils;

namespace BMS.Workflows.CustomActivities.Utils
{
    public class WFErrorHandler : IErrorHandler, IServiceBehavior
    {
        public void AddBindingParameters(ServiceDescription serviceDescription, ServiceHostBase serviceHostBase,
                Collection<ServiceEndpoint> endpoints, BindingParameterCollection bindingParameters)
        {
        }

        public void ApplyDispatchBehavior(ServiceDescription serviceDescription, ServiceHostBase serviceHostBase)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                IErrorHandler errorHandler = new WFErrorHandler();

                foreach (ChannelDispatcherBase channelDispatcherBase in serviceHostBase.ChannelDispatchers)
                {
                    ChannelDispatcher channelDispatcher = channelDispatcherBase as ChannelDispatcher;

                    if (channelDispatcher != null)
                    {
                        channelDispatcher.ErrorHandlers.Add(errorHandler);
                    }
                }
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        public bool HandleError(Exception error)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                Tracer.TraceException(error);
                return true;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


        public void Validate(ServiceDescription serviceDescription, ServiceHostBase serviceHostBase)
        {
        }

        public void ProvideFault(Exception error, MessageVersion version, ref Message fault)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                WFInvalidOperation wfFault = null;
                FaultException<GenericWFServiceFault> genericFault = null;
                WfOtherException otherEx = null;
                MessageFault mf = null;

                if ((error as System.OperationCanceledException) != null)
                {
                    if (error.InnerException != null && (error.InnerException as System.Runtime.DurableInstancing.InstancePersistenceCommandException) != null)
                    {
                        if (error.InnerException.Message.Contains(Resources.WF_ERROR_INSTANCE_PART))
                        {
                            wfFault = new WFInvalidOperation()
                            {
                                ErrorMessage = GetExceptionMessage(error),
                                StackTrace = GetStackTrace(error),
                                FriendlyMessage = GetFriendlyMessage(error)
                            };

                            genericFault = new FaultException<GenericWFServiceFault>(new GenericWFServiceFault() { ServiceFault = wfFault });
                            mf = genericFault.CreateMessageFault();
                            fault = Message.CreateMessage(version, mf, genericFault.Action);
                        }
                        return;
                    }
                }

                otherEx = new WfOtherException() { ErrorMessage = GetExceptionMessage(error), StackTrace = GetStackTrace(error) };
                genericFault = new FaultException<GenericWFServiceFault>(new GenericWFServiceFault() { ServiceFault = otherEx });
                mf = genericFault.CreateMessageFault();
                fault = Message.CreateMessage(version, mf, genericFault.Action);
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal string GetFriendlyMessage(Exception ex) 
        {
            StackTrace stackTrace = new StackTrace(ex, false);

            foreach (StackFrame frame in stackTrace.GetFrames())
            {
                System.Reflection.MethodBase method = frame.GetMethod();
                if (method.DeclaringType == null)
                    continue;
                if (method.DeclaringType.FullName == typeof(BMS.ServicesWrapper.Proxy.WF.BMSWorkflowsWFIBedUnavailableFlowClient).FullName)
                    return Resources.WF_INVALID_OPERATION_BED;
            }
            return Resources.WF_INVALID_OPERATION;
        }

        internal string GetExceptionMessage(Exception ex)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                StringBuilder sb = new StringBuilder();
                if (ex != null)
                {
                    while (ex != null)
                    {
                        sb.AppendLine(ex.Message);
                        ex = ex.InnerException;
                    }
                }
                return sb.ToString();
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }

        internal string GetStackTrace(Exception ex)
        {
            DateTime entryInLogMethodTime = DateTime.UtcNow;
            if (InfoWorld.Tracing.IWTrace.IsEntryEnabled)
            {
                InfoWorld.Tracing.IWTrace.Entry(System.Reflection.MethodBase.GetCurrentMethod(), entryInLogMethodTime);
            }
            try
            {
                if (ex != null)
                {
                    if (String.IsNullOrEmpty(ex.StackTrace))
                        return ex.InnerException == null ? String.Empty : ex.InnerException.StackTrace;

                    return ex.StackTrace;
                }
                return string.Empty;
            }
            finally
            {
                if (InfoWorld.Tracing.IWTrace.IsExitEnabled)
                {
                    InfoWorld.Tracing.IWTrace.Exit(System.Reflection.MethodBase.GetCurrentMethod(), DateTime.UtcNow, entryInLogMethodTime);
                }
            }
        }


    }
}